import Content from "../models/Content.js";
import Transaction from "../models/Transaction.js";
import {db} from "../config/database.js"
// Helper untuk membuat slug acak sepanjang 5 karakter
const generateSlug = () => {
  const chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
  let slug = "";
  for (let i = 0; i < 5; i++) {
    slug += chars.charAt(Math.floor(Math.random() * chars.length));
  }
  return slug;
};

export const addContent = async (req, res) => {
  try {
    const { title, content, vip_content, price } = req.body;
    // Validasi input sederhana
    if (!title || !content || !vip_content) {
      return res.status(400).json({ msg: "Field judul dan konten harus diisi!" });
    }
    // Buat slug unik
    let slug;
    let isUnique = false;
    
    // Loop hingga dapat slug unik
    while (!isUnique) {
      slug = generateSlug();
      const existing = await Content.findOne({ where: { slug } });
      if (!existing) isUnique = true;
    }
    // Simpan konten baru
    const newContent = await Content.create({
      slug,
      title,
      content,
      vip_content,
      price: price || 1500
    });
    return res.status(201).json({
      msg: "Konten berhasil ditambahkan",
      data: newContent,
    });
  } catch (error) {
    console.error("❌ Error tambah konten:", error);
    return res.status(500).json({ msg: "Terjadi kesalahan server" });
  }
};

export const renderContent = async (req, res) => {
  try {
    // Ambil slug dari params
    const { slug } = req.params;
    
    // Cari konten berdasarkan slug
    const getSlug = await Content.findOne({ where: { slug } });
    
    if (!getSlug) {
      return res.render("./pages/content", {
        title: "Halaman tidak ditemukan",
        data: { message: "Halaman tidak ditemukan" },
      });
    }
    
    // Data yang dikirim ke EJS
    const data = {
      slug: getSlug.slug,
      judul: getSlug.title,
      content: getSlug.content,
      price: getSlug.price,
    };
    
    return res.render("./pages/content", {
      title: getSlug.title,
      data,
    });
  } catch (error) {
    console.error(error);
    res.status(500).render("./pages/content", {
      title: "Error",
      data: { message: "Terjadi kesalahan server" },
    });
  }
};

const apiKey = "OpvIdiD5PDTCEIaHCGY81ZA0wf6XxE41";

export const createPayment = async (req, res) => {
  const t = await db.transaction();
  try {
    const { slug } = req.body;
    
    // 🔸 Cek data konten
    const data = await Content.findOne({ where: { slug }, transaction: t });
    
    if (!data) {
      await t.rollback();
      return res.status(404).json({ msg: "Pembelian gagal, konten tidak ditemukan" });
    }
    
    // 🔸 Generate order_id unik
    const order_id = `RIZ-${Date.now()}`;
    
    // 🔸 Kirim request ke payment gateway
    const response = await fetch("https://app.pakasir.com/api/transactioncreate/qris", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({
        project: "gilang",
        order_id,
        amount: data.price,
        api_key: apiKey,
      }),
    });
    
    const result = await response.json();
    // Jika gagal dari API
    if (!response.ok || !result?.payment) {
      await t.rollback();
      return res.status(500).json({
        msg: "Gagal membuat pembayaran",
        detail: result,
      });
    }
    
    // 🔸 Simpan transaksi ke DB
    await Transaction.create(
      {
        order_id,
        slug,
        amount: data.price,
        status: "pending",
      },
      { transaction: t }
    );
    
    // 🔹 Commit transaksi kalau semua sukses
    await t.commit();
    return res.status(200).json({
      msg: "Pembayaran berhasil dibuat",
      payment: result.payment,
    });
  } catch (error) {
    console.error("❌ Error createPayment:", error);
    
    // 🔹 Rollback jika ada error
    await t.rollback();
    
    return res.status(500).json({ msg: "Terjadi kesalahan server" });
  }
};

export const cekPayment = async (req, res) => {
  try {
    const { order_id, amount } = req.body;
    // ✅ Validasi input
    if (!order_id || !amount) {
      return res.status(400).json({ msg: "order_id dan amount wajib diisi" });
    }
    
    // ✅ Cek transaksi di database
    const dataTransaction = await Transaction.findOne({ where: { order_id } });
    if (!dataTransaction) {
      return res.status(404).json({ msg: "Transaksi tidak ditemukan di database" });
    }
    
    // 🔹 Gunakan nilai amount dari database (lebih akurat)
    const amountToCheck = dataTransaction.amount || amount;
    
    // 🔹 Request ke API Pakasir
    const response = await fetch(
      `https://app.pakasir.com/api/transactiondetail?project=gilang&amount=${amountToCheck}&order_id=${order_id}&api_key=${apiKey}`,
      { method: "GET" }
    );

    const result = await response.json();

    // ✅ Validasi hasil respon
    if (!response.ok || !result.transaction) {
      return res.status(response.status).json({
        msg: "Gagal memeriksa status transaksi",
        detail: result,
      });
    }

    const { status } = result.transaction;
    console.log(`🧾 Transaksi ${order_id} status dari Pakasir: ${status}`);

    // 🔹 Jika masih pending
    if (status !== "completed") {
      return res.status(200).json({ status: "pending" });
    }

    // ✅ Update status transaksi jika belum completed
    if (dataTransaction.status !== "completed") {
      dataTransaction.status = "completed";
      await dataTransaction.save();
      console.log(`✅ Status transaksi ${order_id} diperbarui ke "completed"`);
    }
    
    // ✅ Ambil konten VIP dari database
    const dataContent = await Content.findOne({
      where: { slug: dataTransaction.slug },
    });
    
    if (!dataContent) {
      return res.status(404).json({ msg: "Konten tidak ditemukan" });
    }

    // 🔹 Kirim respon sukses
    return res.status(200).json({
      status: "completed",
      vip_content: dataContent.vip_content,
    });

  } catch (error) {
    console.error("❌ Error cekPayment:", error);
    return res.status(500).json({
      msg: "Terjadi kesalahan server",
      error: error.message,
    });
  }
};